%%%%%%%%% TYPE 1 SINGULARITY LOCI OF ORIENTATIONAL PARALLEL MECHANISMS %%%%%%%%
%
% This program plots the Type 1 singularity loci of a symmetrical
% spherical parallel mechanism for a constant torsion (sigma). The program
% calls the subprogram SLDraw.m. The program ExportESP exports the workspace 
% into an EPS file.
%
% Last modified on June 16, 2005
% Written by Ilian A. Bonev
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

 clear all;

%INITIAL PARAMETERS
 beta1 = 45*pi/180;   %angle of the base axes with the negative base z-axis
 beta2 = 45*pi/180;   %angle of the platform axes with the mobile z-axis
 alpha_min = 0*pi/180;  %minimum leg length
 alpha_max = 180*pi/180; %maximum leg length
 sigma = 0;           %the constant torsion angle
 hatch = 1;           %default value for workspace hatch (1 = on, -1 = off)

%GRAPHICS INITIALIZATION
 fig = figure('Name', 'Type 1 Singularity Loci', 'NumberTitle', 'off',...
              'Position', [90,70,850,600]);
 set(fig,'BackingStore', 'off');
 set(fig,'Color','white');
 set(gca, 'drawmode', 'fast');
 subplot('Position', [0.08,0.1,0.6,0.85]);
 hold on; axis equal; axis([-1.1*pi 1.1*pi -1.1*pi 1.1*pi]); axis off
 frange = 0:0.05:2*pi; frange(end+1) = frange(1);
 plot(pi*cos(frange),pi*sin(frange),'k-','LineWidth',1);
 text(0.1,1.03*pi,'\theta=\pi');
 text(0.1,1.05*pi/2,'\theta=\pi/2');
 plot(0.5*pi*cos(frange),0.5*pi*sin(frange),'k-','LineWidth',1);
 for i = 1:3,
   point_i(i) = line('color',[0,0,1],'LineStyle','none','Marker','.',...
                     'erasemode','xor','MarkerSize',12);
   point_e(i) = line('color',[0,0,1],'LineStyle','none','Marker','.',...
                     'erasemode','xor','MarkerSize',12);
   curve_iL(i) = line('color',[0.5,0.5,0.5],'LineStyle','-',...
                      'erasemode','xor','LineWidth',1);
   curve_iB(i) = line('color',[0,0,1],'LineStyle','-','erasemode','xor',...
                      'LineWidth',1);
   curve_eL(i) = line('color',[0.5,0.5,0.5],'LineStyle','-',...
                      'erasemode','xor','LineWidth',1);
   curve_eB(i) = line('color',[0,0,1],'LineStyle','-','erasemode','xor',...
                      'LineWidth',1);
 end
 HRegion = line('color',[1,0,0],'LineStyle','none','Marker','.',...
                'erasemode','xor','MarkerSize',1);

%DEFINE THE <QUIT> MENU
 uimenu(fig,'Label','Quit','CallBack','close(fig); clear all');
 
%DEFINE THE <EXPORT> MENU
 uimenu(fig,'Label','Export','CallBack','ExportEPS');
            
%DEFINE THE beta1 SLIDER
 sli_beta1 = uicontrol(fig,'Style','slider','Pos',[670 300 120 17],...
   'Min',0,'Max',+90,'SliderStep',[1/90 1/9],...
   'Val',beta1*180/pi,'CallBack',['beta1=(pi/180)*round(get(sli_beta1,''Val''));',...
     'set(beta1_cur,''String'',[num2str(round(180*beta1/pi)),'''']); SLDraw']);

%DEFINE THE beta2 SLIDER
 sli_beta2 = uicontrol(fig,'Style','slider','Pos',[670 250 120 17],...
   'Min',+10,'Max',+90,'SliderStep',[1/80 1/8],...
   'Val',beta2*180/pi,'CallBack',['beta2=(pi/180)*round(get(sli_beta2,''Val''));',...
     'set(beta2_cur,''String'',[num2str(round(180*beta2/pi)),'''']); SLDraw']);

%DEFINE THE alpha_min SLIDER
 sli_alpha_min = uicontrol(fig,'Style','slider','Pos',[670 200 120 17],...
   'Min',0,'Max',180,'SliderStep',[1/180 10/180],'Val',alpha_min*180/pi,...
   'CallBack',['alpha_min=(pi/180)*round(get(sli_alpha_min,''Val''));',...
     'set(alpha_min_cur,''String'',[num2str(round(180*alpha_min/pi)),'''']);'...
     'SLDraw']);

%DEFINE THE alpha_max SLIDER
 sli_alpha_max = uicontrol(fig,'Style','slider','Pos',[670 150 120 17],...
   'Min',0,'Max',180,'SliderStep',[1/180 10/180],'Val',alpha_max*180/pi,...
   'CallBack',['alpha_max=(pi/180)*round(get(sli_alpha_max,''Val''));',...
     'set(alpha_max_cur,''String'',[num2str(round(180*alpha_max/pi)),'''']);'...
     'SLDraw']);

%DEFINE THE sigma SLIDER
 sli_sigma = uicontrol(fig,'Style','slider','Pos',[670 100 120 17],...
   'Min',-180,'Max',180,'SliderStep',[1/360 10/360],...
   'Val',0,'CallBack',['sigma=(pi/180)*round(get(sli_sigma,''Val''));',...
     'set(sigma_cur,''String'',[num2str(round(180*sigma/pi)),'''']); SLDraw']);

%DEFINE TEXT CONTROL FOR THE MINIMUM VALUES
 uicontrol(fig,'Style','text','Pos',[624 300 45 17],'String','0');
 uicontrol(fig,'Style','text','Pos',[624 250 45 17],'String','10');
 uicontrol(fig,'Style','text','Pos',[624 200 45 17],'String','0');
 uicontrol(fig,'Style','text','Pos',[624 150 45 17],'String','0');
 uicontrol(fig,'Style','text','Pos',[624 100 45 17],'String','-180');
   

%DEFINE TEXT CONTROL FOR THE MAXIMUM VALUES
 uicontrol(fig,'Style','text','Pos',[791 300 45 17],'String','+90');
 uicontrol(fig,'Style','text','Pos',[791 250 45 17],'String','+90');
 uicontrol(fig,'Style','text','Pos',[791 200 45 17],'String','+180');
 uicontrol(fig,'Style','text','Pos',[791 150 45 17],'String','+180');
 uicontrol(fig,'Style','text','Pos',[791 100 45 17],'String','+180');

%DEFINE THE SLIDER LABELS
 uicontrol(fig,'Style','text','Pos',[700 320 20 17],'FontSize',10,...
     'HorizontalAlignment','left','FontName','Symbol','String','b1:');
 uicontrol(fig,'Style','text','Pos',[700 270 20 17],'FontSize',10,...
     'HorizontalAlignment','left','FontName','Symbol','String','b2:');
 uicontrol(fig,'Style','text','Pos',[680 220 15 17],'FontSize',10,...
     'HorizontalAlignment','left','FontName','Symbol','String','a');
 uicontrol(fig,'Style','text','Pos',[688 220 32 17],'FontSize',10,...
     'HorizontalAlignment','left','String','_min:');
 uicontrol(fig,'Style','text','Pos',[677 170 15 17],'FontSize',10,...
     'HorizontalAlignment','left','FontName','Symbol','String','a');
 uicontrol(fig,'Style','text','Pos',[685 170 35 17],'FontSize',10,...
     'HorizontalAlignment','left','String','_max:');
 uicontrol(fig,'Style','text','Pos',[700 120 20 17],'FontSize',10,...
     'HorizontalAlignment','left','FontName','Symbol','String','s:');
 
%DEFINE THE TEXT CONTROLS FOR THE CURRENT VALUES
 beta1_cur = uicontrol(fig,'Style','text','Pos',[722 320 45 17],...
   'String',[num2str(round(get(sli_beta1,'Value'))),'']);
 beta2_cur = uicontrol(fig,'Style','text','Pos',[722 270 45 17],...
   'String',[num2str(round(get(sli_beta2,'Value'))),'']);
 alpha_min_cur = uicontrol(fig,'Style','text','Pos',[722 220 45 17],...
   'String',[num2str(round(get(sli_alpha_min,'Value'))),'']);
 alpha_max_cur = uicontrol(fig,'Style','text','Pos',[722 170 45 17],...
   'String',[num2str(round(get(sli_alpha_max,'Value'))),'']);
 sigma_cur = uicontrol(fig,'Style','text','Pos',[722 120 45 17],...
   'String',[num2str(round(get(sli_sigma,'Value'))),'']);

%DEFINE THE WORKSPACE HATCH CHECKBOX
 uicontrol(fig,'Style','checkbox','Pos',[675 360 110 20],'String',...
   'Workspace Hatch','Value',1,'Callback','hatch=-1*hatch;SLDraw;');

 SLDraw;